function AnimalMoveEvent.new(sourceObject, targetObject, animals, moveType)

	local event = AnimalMoveEvent.emptyNew()

	event.sourceObject = sourceObject
	event.targetObject = targetObject
	event.animals = animals
	event.moveType = moveType

	return event

end


function AnimalMoveEvent:readStream(streamId, connection)

	if connection:getIsServer() then

		self.errorCode = streamReadUIntN(streamId, 3)

	else

		self.moveType = streamReadString(streamId)

		self.sourceObject = NetworkUtil.readNodeObject(streamId)
		self.targetObject = NetworkUtil.readNodeObject(streamId)

		local numAnimals = streamReadUInt16(streamId)

		self.animals = {}

		for i = 1, numAnimals do

			local animal = Animal.new()
			local success = animal:readStream(streamId, connection)
			table.insert(self.animals, animal)

		end

	end

	self:run(connection)

end


function AnimalMoveEvent:writeStream(streamId, connection)

	if not connection:getIsServer() then
		streamWriteUIntN(streamId, self.errorCode, 3)
		return
	end

	streamWriteString(streamId, self.moveType)

	NetworkUtil.writeNodeObject(streamId, self.sourceObject)
	NetworkUtil.writeNodeObject(streamId, self.targetObject)

	streamWriteUInt16(streamId, #self.animals)

	for _, animal in pairs(self.animals) do local success = animal:writeStream(streamId, connection) end

end


function AnimalMoveEvent:run(connection)

	if connection:getIsServer() then

		g_messageCenter:publish(AnimalMoveEvent, self.errorCode)
		return

	end

	local userId = g_currentMission.userManager:getUniqueUserIdByConnection(connection)
	local farmId = g_farmManager:getFarmForUniqueUserId(userId).farmId

	for _, animal in pairs(self.animals) do

		local errorCode = AnimalMoveEvent.validate(self.sourceObject, self.targetObject, farmId, animal.subTypeIndex)

		if errorCode ~= nil then
			connection:sendEvent(AnimalMoveEvent.newServerToClient(errorCode))
			return
		end
	
	end

	local clusterSystemSource = self.sourceObject:getClusterSystem()
	local clusterSystemTarget = self.targetObject:getClusterSystem()

	-- Compatibility: some targets (e.g. ExtendedProductionPoint / Butcher Table style placeables)
	-- are not real husbandries. In those cases we must avoid RL-specific ID munging and use a robust
	-- cluster removal strategy that works with both vanilla and RL cluster systems.
	local function isExtendedProductionTarget(obj)
		return obj ~= nil and obj.animalSubTypeToFillType ~= nil and obj.storage ~= nil
	end

	local function removeFromClusterSystem(cs, cluster)
		if cs == nil or cluster == nil then
			return
		end

		-- If the cluster system stores animals in an array, prefer index-based removal
		local animalsArr = cs.animals
		if type(animalsArr) == "table" then
			local idx = nil
			for i, a in ipairs(animalsArr) do
				if a == cluster then
					idx = i
					break
				end
				-- fallback match by uniqueId + birthday country if present
				if idx == nil and a ~= nil and a.uniqueId ~= nil and cluster.uniqueId ~= nil and tostring(a.uniqueId) == tostring(cluster.uniqueId) then
					local ac = a.birthday ~= nil and a.birthday.country or nil
					local cc = cluster.birthday ~= nil and cluster.birthday.country or nil
					if ac == cc then
						idx = i
						break
					end
				end
			end

			if idx ~= nil then
				cs:removeCluster(idx)
				return
			end
		end

		-- Otherwise try common identifier fields
		local id = cluster.idFull or cluster.id or cluster.uniqueId
		if id ~= nil then
			cs:removeCluster(id)
		end
	end

	local extendedTarget = isExtendedProductionTarget(self.targetObject)

	for _, animal in pairs(self.animals) do
		removeFromClusterSystem(clusterSystemSource, animal)

		-- RL sets id/idFull to nil for some move flows; this breaks mods that expect stable animal ids.
		-- Keep ids intact for ExtendedProduction targets (Butcher Table etc.).
		if not extendedTarget then
			animal.id, animal.idFull = nil, nil
		end

		-- For ExtendedProductionPoint style targets (e.g. Butcher Table), the placeable implements addCluster() to convert animals into fill levels.
		-- RL must call targetObject:addCluster() (not the cluster system), otherwise animals disappear without producing output.
		if extendedTarget and type(self.targetObject.addCluster) == "function" then
			self.targetObject:addCluster(animal)
		else
			clusterSystemTarget:addCluster(animal)
		end
	end


	connection:sendEvent(AnimalMoveEvent.newServerToClient(AnimalMoveEvent.MOVE_SUCCESS))

	if g_server ~= nil and not g_server.netIsRunning then return end

	local husbandry, trailer
	
	if self.moveType == "SOURCE" then 
		husbandry, trailer = self.sourceObject, self.targetObject
	else
		husbandry, trailer = self.targetObject, self.sourceObject
	end

	if #self.animals == 1 then
        husbandry:addRLMessage(string.format("MOVE_ANIMALS_%s_SINGLE", self.moveType), nil, { trailer:getName() })
    elseif #self.animals > 0 then
        husbandry:addRLMessage(string.format("MOVE_ANIMALS_%s_MULTIPLE", self.moveType), nil, { #self.animals, trailer:getName() })
    end

end


function AnimalMoveEvent.validate(sourceObject, targetObject, farmId, subTypeIndex)

	if sourceObject == nil then return AnimalMoveEvent.MOVE_ERROR_SOURCE_OBJECT_DOES_NOT_EXIST end
	
	if targetObject == nil then return AnimalMoveEvent.MOVE_ERROR_TARGET_OBJECT_DOES_NOT_EXIST end

	if not g_currentMission.accessHandler:canFarmAccess(farmId, sourceObject) or not g_currentMission.accessHandler:canFarmAccess(farmId, targetObject) then return AnimalMoveEvent.MOVE_ERROR_NO_PERMISSION end

	if not targetObject:getSupportsAnimalSubType(subTypeIndex) then return AnimalMoveEvent.MOVE_ERROR_ANIMAL_NOT_SUPPORTED end

	if targetObject:getNumOfFreeAnimalSlots(subTypeIndex) < 1 then return AnimalMoveEvent.MOVE_ERROR_NOT_ENOUGH_SPACE end

	return nil

end