-- FS25_NoMassTensionBelts / scripts/NoMassTensionBelts.lua
-- Global script that patches TensionBelts so fastening belts does NOT change mass/inertia
-- and uses milder joint springs to reduce jitter/shaking on loaded objects.
-- Drop this mod ZIP into your mods folder; no XML edits required.

NoMassTensionBelts = {}
NoMassTensionBelts.installed = false

local function dbg(msg, ...)
    print(("[NoMassTensionBelts] " .. msg):format(...))
end

function NoMassTensionBelts:install()
    if self.installed then return end
    if TensionBelts == nil then
        -- Giants hasn't created the class yet; try again on next update
        return
    end

    -- Keep originals in case someone wants to restore (not used here but handy for debugging)
    self._orig_lockTensionBeltObject   = TensionBelts.lockTensionBeltObject
    self._orig_freeTensionBeltObject   = TensionBelts.freeTensionBeltObject
    self._orig_getAdditionalComponentMass = TensionBelts.getAdditionalComponentMass

    ----------------------------------------------------------------------
    -- Patch: lockTensionBeltObject
    -- Vanilla reduces mass (setMass 0.01), boosts inertia on split shapes,
    -- and/or toggles reduced component mass on objects, which can change how
    -- the loaded vehicle feels and often increases jitter. We avoid all that.
    ----------------------------------------------------------------------
    function TensionBelts:lockTensionBeltObject(objectId, objectsToJointTable, isDynamic, jointNode, object)
        if objectsToJointTable[objectId] ~= nil then
            return
        end

        -- Always use a dynamic joint with softer springs. No mass/inertia tweaks.
        if self.isServer then
            local ctor = JointConstructor.new()
            ctor:setActors(jointNode, objectId)

            local jointTransform = createTransformGroup("tensionBeltJoint")
            link(jointNode, jointTransform)
            local wx, wy, wz = getWorldTranslation(objectId)
            setWorldTranslation(jointTransform, wx, wy, wz)

            ctor:setJointTransforms(jointTransform, jointTransform)
            ctor:setEnableCollision(true)

            -- Fully locked rotation like vanilla, but with softer springs to reduce oscillations.
            ctor:setRotationLimit(0, 0, 0)
            ctor:setRotationLimit(1, 0, 0)
            ctor:setRotationLimit(2, 0, 0)

            -- Softer spring/damping than default (which is 1000/10 in vanilla).
            -- Feel free to tweak if needed for your map/loads.
            ctor:setRotationLimitSpring(200, 5, 200, 5, 200, 5)
            ctor:setTranslationLimitSpring(200, 5, 200, 5, 200, 5)

            local jointIndex = ctor:finalize()

            -- IMPORTANT: We do NOT call setMass(0.01), setReducedComponentMass(true),
            -- setInertiaScale(20,20,20) or setRigidBodyType(KINEMATIC). We also don't
            -- change dynamic mount type. This keeps the object's original physics.
            objectsToJointTable[objectId] = {
                jointIndex = jointIndex,
                jointTransform = jointTransform,
                object = object
            }

            if object ~= nil and object.setCanBeSold ~= nil then
                object:setCanBeSold(false)
            end

            if getSplitType(objectId) ~= 0 then
                setUserAttribute(objectId, "isTensionBeltMounted", UserAttributeType.BOOLEAN, true)
                g_messageCenter:publish(MessageType.TREE_SHAPE_MOUNTED, objectId, self)
            end
        else
            -- Client mirror entry
            objectsToJointTable[objectId] = { jointIndex = 0, object = object }
        end
    end

    ----------------------------------------------------------------------
    -- Patch: freeTensionBeltObject
    -- Matches the behavior of removing just the joint/transform—no mass restore
    -- needed because we never changed it. Also re‑enables selling.
    ----------------------------------------------------------------------
    function TensionBelts:freeTensionBeltObject(objectId, objectsToJointTable, isDynamic, object)
        if entityExists(objectId) then
            local jointData = objectsToJointTable[objectId]
            if self.isServer and jointData ~= nil and jointData.jointIndex ~= nil then
                removeJoint(jointData.jointIndex)
                delete(jointData.jointTransform)
            end

            if object ~= nil and object.setCanBeSold ~= nil then
                object:setCanBeSold(true)
            end

            if getSplitType(objectId) ~= 0 then
                setUserAttribute(objectId, "isTensionBeltMounted", UserAttributeType.BOOLEAN, false)
            end
        end

        objectsToJointTable[objectId] = nil
    end

    ----------------------------------------------------------------------
    -- Patch: getAdditionalComponentMass → always 0
    -- Ensures vehicles don't gain extra "virtual" mass from mounted objects.
    ----------------------------------------------------------------------
    function TensionBelts:getAdditionalComponentMass(superFunc, component)
        return 0
    end

    self.installed = true
    dbg("Installed patches (no mass/inertia change, softer springs).")
end

-- Mod event hooks
function NoMassTensionBelts:loadMap()   self:install() end
function NoMassTensionBelts:deleteMap() end
function NoMassTensionBelts:update(dt)  self:install() end -- ensure late install if Giants loads TensionBelts later

addModEventListener(NoMassTensionBelts)
